<?php
session_start();
include('include/config.php');
error_reporting(E_ALL);
ini_set('display_errors', 1);

// ============================
// Fetch Medicines
// ============================
$medicines = [];
$medRes = mysqli_query($con, "SELECT id, name FROM medicine ORDER BY name ASC");
while ($row = mysqli_fetch_assoc($medRes)) {
    $medicines[] = $row;
}

// ============================
// Fetch Departments
// ============================
$departments = [];
$deptRes = mysqli_query($con, "SELECT id, dname FROM departments ORDER BY dname ASC");
while ($row = mysqli_fetch_assoc($deptRes)) {
    $departments[] = $row;
}

// ============================
// Handle Stock Transfer
// ============================
$alertHtml = "";
if (isset($_POST['submit'])) {
    $medicine_id     = (int) $_POST['medicine_id'];
    $from_department = (int) $_POST['from_department'];
    $to_department   = (int) $_POST['to_department'];
    $quantity        = (int) $_POST['quantity'];

    if ($medicine_id <= 0 || $from_department <= 0 || $to_department <= 0 || $quantity <= 0) {
        $alertHtml = "<div class='alert alert-danger'>❌ All fields required and quantity must be positive.</div>";
    } elseif ($from_department === $to_department) {
        $alertHtml = "<div class='alert alert-danger'>❌ Source and destination cannot be the same.</div>";
    } else {
        // Check stock
        $stmt = $con->prepare("SELECT stock_quantity FROM department_medicine_stock WHERE department_id=? AND medicine_id=?");
        $stmt->bind_param("ii", $from_department, $medicine_id);
        $stmt->execute();
        $stmt->bind_result($currentStock);
        if ($stmt->fetch()) {
            if ($currentStock < $quantity) {
                $alertHtml = "<div class='alert alert-danger'>❌ Not enough stock. Available: $currentStock</div>";
            } else {
                $stmt->close();
                $con->begin_transaction();

                // Deduct from source
                $upd1 = $con->prepare("UPDATE department_medicine_stock SET stock_quantity=stock_quantity-? WHERE department_id=? AND medicine_id=?");
                $upd1->bind_param("iii", $quantity, $from_department, $medicine_id);
                $ok1 = $upd1->execute();
                $upd1->close();

                // Add to destination
                $check = $con->prepare("SELECT id FROM department_medicine_stock WHERE department_id=? AND medicine_id=?");
                $check->bind_param("ii", $to_department, $medicine_id);
                $check->execute();
                $check->store_result();
                if ($check->num_rows > 0) {
                    $check->close();
                    $upd2 = $con->prepare("UPDATE department_medicine_stock SET stock_quantity=stock_quantity+? WHERE department_id=? AND medicine_id=?");
                    $upd2->bind_param("iii", $quantity, $to_department, $medicine_id);
                    $ok2 = $upd2->execute();
                    $upd2->close();
                } else {
                    $check->close();
                    $ins = $con->prepare("INSERT INTO department_medicine_stock (department_id, medicine_id, stock_quantity) VALUES (?,?,?)");
                    $ins->bind_param("iii", $to_department, $medicine_id, $quantity);
                    $ok2 = $ins->execute();
                    $ins->close();
                }

                // Log transfer
                $log = $con->prepare("INSERT INTO stock_transfer_log (medicine_id, from_department_id, to_department_id, quantity, transfer_date) VALUES (?,?,?,?,NOW())");
                $log->bind_param("iiii", $medicine_id, $from_department, $to_department, $quantity);
                $ok3 = $log->execute();
                $log->close();

                if ($ok1 && $ok2 && $ok3) {
                    $con->commit();
                    $alertHtml = "<div class='alert alert-success'>✅ Stock transferred successfully.</div>";
                } else {
                    $con->rollback();
                    $alertHtml = "<div class='alert alert-danger'>❌ Transfer failed, rolled back.</div>";
                }
            }
        } else {
            $alertHtml = "<div class='alert alert-danger'>❌ No stock record found in source department.</div>";
        }
        $stmt->close();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>Stock Transfer</title>
<link rel="stylesheet" href="assets/vendors/mdi/css/materialdesignicons.min.css" />
<link rel="stylesheet" href="assets/vendors/css/vendor.bundle.base.css" />
<link rel="stylesheet" href="assets/css/style.css" />
<link rel="shortcut icon" href="assets/images/favicon.png" />
</head>
<body>
<div class="container-scroller">
    <?php include('include/sidebar.php'); ?>
    <div class="container-fluid page-body-wrapper">
        <?php include('include/header.php'); ?>
        <div class="main-panel">
            <div class="content-wrapper">

                <section id="page-title">
                    <div class="row">
                        <div class="col-sm-8">
                            <h1 class="mainTitle">Stock Transfer</h1>
                        </div>
                    </div>
                </section>

                <div class="container bg-white p-4 rounded shadow-sm">

                    <!-- Alert -->
                    <?= $alertHtml ?>

                    <!-- Transfer Form -->
                    <form method="post">
                        <div class="row">
                            <!-- Medicine -->
                            <div class="col-md-4 mb-3">
                                <label>Medicine</label>
                                <select name="medicine_id" class="form-control" required>
                                    <option value="">-- Select Medicine --</option>
                                    <?php foreach ($medicines as $m): ?>
                                        <option value="<?= $m['id'] ?>"><?= htmlspecialchars($m['name']) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>

                            <!-- From Department -->
                            <div class="col-md-4 mb-3">
                                <label>From Department</label>
                                <select name="from_department" class="form-control" required>
                                    <option value="">-- Select Department --</option>
                                    <?php foreach ($departments as $d): ?>
                                        <option value="<?= $d['id'] ?>"><?= htmlspecialchars($d['dname']) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>

                            <!-- To Department -->
                            <div class="col-md-4 mb-3">
                                <label>To Department</label>
                                <select name="to_department" class="form-control" required>
                                    <option value="">-- Select Department --</option>
                                    <?php foreach ($departments as $d): ?>
                                        <option value="<?= $d['id'] ?>"><?= htmlspecialchars($d['dname']) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>

                            <!-- Quantity -->
                            <div class="col-md-4 mb-3">
                                <label>Quantity</label>
                                <input type="number" name="quantity" class="form-control" required>
                            </div>
                        </div>

                        <div class="text-end">
                            <button type="submit" name="submit" class="btn btn-primary">
                                <i class="mdi mdi-transfer"></i> Transfer
                            </button>
                        </div>
                    </form>
                </div>

                <!-- Transfer Log -->
                <div class="container bg-white mt-4 p-4 rounded shadow-sm">
                    <h5>Recent Stock Transfers</h5>
                    <table class="table table-bordered">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Medicine</th>
                                <th>From Dept</th>
                                <th>To Dept</th>
                                <th>Quantity</th>
                                <th>Date</th>
                            </tr>
                        </thead>
                        <tbody>
                        <?php
                        $res = mysqli_query($con, "
                            SELECT l.id, m.name AS med_name, d1.dname AS from_dept, d2.dname AS to_dept, l.quantity, l.transfer_date
                            FROM stock_transfer_log l
                            JOIN medicine m ON m.id=l.medicine_id
                            JOIN departments d1 ON d1.id=l.from_department_id
                            JOIN departments d2 ON d2.id=l.to_department_id
                            ORDER BY l.transfer_date DESC LIMIT 10
                        ");
                        if (mysqli_num_rows($res) == 0) {
                            echo "<tr><td colspan='6' class='text-center'>No transfers yet.</td></tr>";
                        } else {
                            $i=1;
                            while($r=mysqli_fetch_assoc($res)) {
                                echo "<tr>
                                    <td>{$i}</td>
                                    <td>".htmlspecialchars($r['med_name'])."</td>
                                    <td>".htmlspecialchars($r['from_dept'])."</td>
                                    <td>".htmlspecialchars($r['to_dept'])."</td>
                                    <td>{$r['quantity']}</td>
                                    <td>{$r['transfer_date']}</td>
                                </tr>";
                                $i++;
                            }
                        }
                        ?>
                        </tbody>
                    </table>
                </div>

            </div>
            <?php include('include/footer.php'); ?>
        </div>
    </div>
</div>
<script src="assets/vendors/js/vendor.bundle.base.js"></script>
</body>
</html>
